"use client";

import { useEffect, useState } from "react";
import { AnnonceProps } from "@/lib/types/AnnonceProps";
import { useRouter } from "next/navigation";
import Dropzone from "react-dropzone";

export default function EditAnnonceMainImg({ annonce }: { annonce: AnnonceProps }) {
    if (!annonce) {
        return (
            <div className="p-6 text-center">
                <h2 className="text-xl font-semibold">Annonce introuvable</h2>
            </div>
        );
    }
    const router = useRouter();
    const [photoPrincipaleFile, setphotoPrincipaleFile] = useState<File | null>(null);
    const [previewUrl, setPreviewUrl] = useState<string | null>(null);
    const [uploading, setUploading] = useState(false);
    const [imgError, setImgError] = useState<string | null>(null);

    useEffect(() => {
        if (annonce.mainImg) {
            setPreviewUrl(`data:image/jpeg;base64,${annonce.mainImg}`);
        }
    }, [annonce.mainImg]);

    async function handleUploadImage() {
        if (!photoPrincipaleFile) return;
        setImgError(null);
        setUploading(true);
        try {
            const fd = new FormData();
            fd.append("imagePrincipale", photoPrincipaleFile);

            const res = await fetch(`/api/annonces/${annonce.id}`, {
                method: "PATCH",
                body: fd,
            });

            if (!res.ok) {
                const err = await res.json().catch(() => ({}));
                throw new Error(err.message || "Échec de la mise à jour de l'image.");
            }
            router.push(`/annonces/${annonce.id}`); 
        } catch (e: any) {
            setImgError(e.message ?? "Erreur inconnue");
        } finally {
            setUploading(false);
        }
    }

    function onDrop(files: File[]) {
        const f = files[0];
        if (!f) return;
        setphotoPrincipaleFile(f);
        setPreviewUrl(URL.createObjectURL(f));
    }

    return (
        <div className="flex min-h-screen flex-col items-center justify-center bg-gray-50 p-4">
            <div className="mx-auto w-full max-w-md">
                <div className="rounded-xl border border-gray-200 bg-white p-8 shadow-sm">
                    <div className="mt-8">
                        <label className="mb-2 block text-sm font-medium text-gray-700">Image principale</label>
                        <Dropzone
                            multiple={false}
                            onDrop={onDrop}
                            accept={{
                                "image/png": [".png"],
                                "image/jpeg": [".jpg", ".jpeg"],
                                "image/webp": [".webp"],
                            }}
                            maxFiles={1}>
                            {({ getRootProps, getInputProps }) => (
                                <section>
                                    <div
                                        {...getRootProps()}
                                        className="cursor-pointer rounded-md border-2 border-dashed border-gray-300 p-4 text-center hover:bg-gray-100">
                                        <input {...getInputProps()} />
                                        <p>Glissez-déposez votre photo, ou cliquez pour sélectionner.</p>
                                        <p className="text-xs text-gray-500">PNG, JPG, JPEG, WEBP (max 5 Mo)</p>
                                    </div>
                                </section>
                            )}
                        </Dropzone>
                        {previewUrl && (
                            <img
                                src={previewUrl}
                                alt="Image principale"
                                className="mt-4 h-32 w-32 rounded object-cover"
                            />
                        )}{" "}
                        {previewUrl && (
                            <div className="mt-3 flex items-center gap-3">
                                <button
                                    type="button"
                                    onClick={handleUploadImage}
                                    disabled={!photoPrincipaleFile || uploading}
                                    className="rounded-md bg-gray-900 px-4 py-2 text-sm font-medium text-white hover:bg-black disabled:opacity-50">
                                    {uploading ? "Mise à jour…" : "Mettre à jour l’image"}
                                </button>

                                <button
                                    type="button"
                                    onClick={() => {
                                        setphotoPrincipaleFile(null);
                                        setImgError(null);
                                        setPreviewUrl(
                                            annonce.mainImg
                                                ? `data:image/jpeg;base64,${annonce.mainImg}`
                                                : null
                                        );
                                    }}
                                    className="rounded-md border px-4 py-2 text-sm">
                                    Annuler
                                </button>
                            </div>
                        )}
                        {imgError && <p className="mt-2 text-sm text-red-600">{imgError}</p>}
                    </div>
                </div>
            </div>
        </div>
    );
}
